import Database from 'better-sqlite3';

const db = new Database(':memory:', { verbose: console.log });

// Create tables
db.exec(`
  CREATE TABLE IF NOT EXISTS companies (
    id TEXT PRIMARY KEY,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    email TEXT UNIQUE NOT NULL,
    company_name TEXT NOT NULL,
    post_code TEXT NOT NULL,
    business_type TEXT NOT NULL
  );

  CREATE TABLE IF NOT EXISTS suppliers (
    id TEXT PRIMARY KEY,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    name TEXT NOT NULL,
    supplier_code TEXT
  );

  CREATE TABLE IF NOT EXISTS invoices (
    id TEXT PRIMARY KEY,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    company_id TEXT NOT NULL,
    supplier_id TEXT NOT NULL,
    invoice_number TEXT NOT NULL,
    invoice_date DATE NOT NULL,
    supplier_order_number TEXT,
    customer_po_number TEXT,
    scan_date DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (company_id) REFERENCES companies(id),
    FOREIGN KEY (supplier_id) REFERENCES suppliers(id)
  );

  CREATE TABLE IF NOT EXISTS invoice_items (
    id TEXT PRIMARY KEY,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    invoice_id TEXT NOT NULL,
    product_code TEXT NOT NULL,
    product_description TEXT NOT NULL,
    unit_price DECIMAL NOT NULL,
    quantity DECIMAL NOT NULL,
    product_match_id TEXT,
    FOREIGN KEY (invoice_id) REFERENCES invoices(id)
  );

  CREATE INDEX IF NOT EXISTS idx_companies_email ON companies(email);
  CREATE INDEX IF NOT EXISTS idx_invoices_company_id ON invoices(company_id);
  CREATE INDEX IF NOT EXISTS idx_invoices_supplier_id ON invoices(supplier_id);
  CREATE INDEX IF NOT EXISTS idx_invoice_items_invoice_id ON invoice_items(invoice_id);
`);

export interface Company {
  id: string;
  email: string;
  company_name: string;
  post_code: string;
  business_type: string;
  created_at?: string;
}

export interface Supplier {
  id: string;
  name: string;
  supplier_code?: string;
  created_at?: string;
}

export interface Invoice {
  id: string;
  company_id: string;
  supplier_id: string;
  invoice_number: string;
  invoice_date: string;
  supplier_order_number?: string;
  customer_po_number?: string;
  scan_date?: string;
  created_at?: string;
}

export interface InvoiceItem {
  id: string;
  invoice_id: string;
  product_code: string;
  product_description: string;
  unit_price: number;
  quantity: number;
  product_match_id?: string;
  created_at?: string;
}

export const database = {
  // Companies
  createCompany: db.prepare(`
    INSERT INTO companies (id, email, company_name, post_code, business_type)
    VALUES (?, ?, ?, ?, ?)
  `),

  getCompanyByEmail: db.prepare(`
    SELECT * FROM companies WHERE email = ?
  `),

  // Suppliers
  createSupplier: db.prepare(`
    INSERT INTO suppliers (id, name, supplier_code)
    VALUES (?, ?, ?)
  `),

  getSupplierByName: db.prepare(`
    SELECT * FROM suppliers WHERE name = ?
  `),

  // Invoices
  createInvoice: db.prepare(`
    INSERT INTO invoices (
      id, company_id, supplier_id, invoice_number,
      invoice_date, supplier_order_number, customer_po_number
    )
    VALUES (?, ?, ?, ?, ?, ?, ?)
  `),

  getInvoicesByCompanyId: db.prepare(`
    SELECT i.*, s.name as supplier_name
    FROM invoices i
    JOIN suppliers s ON i.supplier_id = s.id
    WHERE i.company_id = ?
    ORDER BY i.created_at DESC
  `),

  // Invoice Items
  createInvoiceItem: db.prepare(`
    INSERT INTO invoice_items (
      id, invoice_id, product_code, product_description,
      unit_price, quantity, product_match_id
    )
    VALUES (?, ?, ?, ?, ?, ?, ?)
  `),

  getInvoiceItemsByInvoiceId: db.prepare(`
    SELECT * FROM invoice_items
    WHERE invoice_id = ?
  `),
};

export function generateId(): string {
  return crypto.randomUUID();
}