import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { useAuth } from '../contexts/AuthContext';
import { Package2, FileText, Users, Building2, AlertCircle } from 'lucide-react';
import InvoiceUpload from '../components/InvoiceUpload';
import InvoiceDataGrid from '../components/InvoiceDataGrid';
import { ExtractedInvoiceData } from '../services/ollama';
import { database, generateId } from '../services/database';

function Dashboard() {
  const { user } = useAuth();
  const navigate = useNavigate();
  const [error, setError] = useState<string | null>(null);
  const [extractedData, setExtractedData] = useState<ExtractedInvoiceData | null>(null);
  const [invoices, setInvoices] = useState<any[]>([]);

  // Redirect if not authenticated
  React.useEffect(() => {
    if (!user) {
      navigate('/auth');
      return;
    }

    // Load invoices for the company
    try {
      const company = database.getCompanyByEmail.get(user.email);
      if (company) {
        const companyInvoices = database.getInvoicesByCompanyId.all(company.id);
        setInvoices(companyInvoices);
      }
    } catch (error) {
      console.error('Error loading invoices:', error);
      setError('Failed to load invoices');
    }
  }, [user, navigate]);

  const handleDataExtracted = (data: ExtractedInvoiceData) => {
    setError(null);
    setExtractedData(data);
  };

  const handleError = (message: string) => {
    setError(message);
    setExtractedData(null);
  };

  const handleSaveInvoice = async (data: ExtractedInvoiceData) => {
    try {
      setError(null);

      // Get company
      const company = database.getCompanyByEmail.get(user?.email);
      if (!company) {
        throw new Error('Company not found');
      }

      // Get or create supplier
      let supplier = database.getSupplierByName.get(data.supplierName);
      if (!supplier) {
        const supplierId = generateId();
        database.createSupplier.run(supplierId, data.supplierName, null);
        supplier = database.getSupplierByName.get(data.supplierName);
      }

      // Create invoice
      const invoiceId = generateId();
      database.createInvoice.run(
        invoiceId,
        company.id,
        supplier.id,
        data.invoiceNumber,
        data.invoiceDate,
        data.supplierOrderNumber || null,
        data.customerPoNumber || null
      );

      // Create invoice items
      for (const item of data.items) {
        database.createInvoiceItem.run(
          generateId(),
          invoiceId,
          item.productCode,
          item.description,
          item.unitPrice,
          item.quantity,
          null
        );
      }

      setExtractedData(null);
      
      // Refresh invoices
      const updatedInvoices = database.getInvoicesByCompanyId.all(company.id);
      setInvoices(updatedInvoices);
    } catch (error) {
      console.error('Error saving invoice:', error);
      setError(error instanceof Error ? error.message : 'Failed to save invoice');
    }
  };

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <header className="bg-white shadow">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
          <h1 className="text-3xl font-bold text-gray-900">Dashboard</h1>
        </div>
      </header>

      {/* Main Content */}
      <main className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {error && (
          <div className="mb-4 p-4 bg-red-50 border border-red-200 rounded-md">
            <div className="flex">
              <AlertCircle className="h-5 w-5 text-red-400" />
              <div className="ml-3">
                <h3 className="text-sm font-medium text-red-800">Error</h3>
                <p className="text-sm text-red-700 mt-1">{error}</p>
              </div>
            </div>
          </div>
        )}

        {!extractedData ? (
          <>
            <div className="grid grid-cols-1 gap-6 sm:grid-cols-2 lg:grid-cols-4">
              {/* Invoices Card */}
              <div className="bg-white overflow-hidden shadow rounded-lg">
                <div className="p-5">
                  <div className="flex items-center">
                    <div className="flex-shrink-0">
                      <FileText className="h-6 w-6 text-gray-400" />
                    </div>
                    <div className="ml-5 w-0 flex-1">
                      <dl>
                        <dt className="text-sm font-medium text-gray-500 truncate">Total Invoices</dt>
                        <dd className="text-lg font-medium text-gray-900">{invoices.length}</dd>
                      </dl>
                    </div>
                  </div>
                </div>
              </div>

              {/* Suppliers Card */}
              <div className="bg-white overflow-hidden shadow rounded-lg">
                <div className="p-5">
                  <div className="flex items-center">
                    <div className="flex-shrink-0">
                      <Users className="h-6 w-6 text-gray-400" />
                    </div>
                    <div className="ml-5 w-0 flex-1">
                      <dl>
                        <dt className="text-sm font-medium text-gray-500 truncate">Suppliers</dt>
                        <dd className="text-lg font-medium text-gray-900">
                          {new Set(invoices.map(i => i.supplier_id)).size}
                        </dd>
                      </dl>
                    </div>
                  </div>
                </div>
              </div>

              {/* Products Card */}
              <div className="bg-white overflow-hidden shadow rounded-lg">
                <div className="p-5">
                  <div className="flex items-center">
                    <div className="flex-shrink-0">
                      <Package2 className="h-6 w-6 text-gray-400" />
                    </div>
                    <div className="ml-5 w-0 flex-1">
                      <dl>
                        <dt className="text-sm font-medium text-gray-500 truncate">Products</dt>
                        <dd className="text-lg font-medium text-gray-900">
                          {invoices.length > 0 ? '...' : '0'}
                        </dd>
                      </dl>
                    </div>
                  </div>
                </div>
              </div>

              {/* Company Info Card */}
              <div className="bg-white overflow-hidden shadow rounded-lg">
                <div className="p-5">
                  <div className="flex items-center">
                    <div className="flex-shrink-0">
                      <Building2 className="h-6 w-6 text-gray-400" />
                    </div>
                    <div className="ml-5 w-0 flex-1">
                      <dl>
                        <dt className="text-sm font-medium text-gray-500 truncate">Company</dt>
                        <dd className="text-lg font-medium text-gray-900 truncate">
                          {user?.email || 'Not Available'}
                        </dd>
                      </dl>
                    </div>
                  </div>
                </div>
              </div>
            </div>

            <div className="mt-8">
              <div className="bg-white shadow rounded-lg">
                <div className="px-4 py-5 sm:p-6">
                  <h2 className="text-lg leading-6 font-medium text-gray-900 mb-4">Upload Invoice</h2>
                  <InvoiceUpload
                    onDataExtracted={handleDataExtracted}
                    onError={handleError}
                  />
                </div>
              </div>
            </div>

            {/* Recent Invoices Table */}
            <div className="mt-8">
              <div className="bg-white shadow rounded-lg">
                <div className="px-4 py-5 sm:p-6">
                  <h2 className="text-lg leading-6 font-medium text-gray-900">Recent Invoices</h2>
                  <div className="mt-4">
                    <div className="flex flex-col">
                      <div className="-my-2 overflow-x-auto sm:-mx-6 lg:-mx-8">
                        <div className="py-2 align-middle inline-block min-w-full sm:px-6 lg:px-8">
                          <div className="shadow overflow-hidden border-b border-gray-200 sm:rounded-lg">
                            <table className="min-w-full divide-y divide-gray-200">
                              <thead className="bg-gray-50">
                                <tr>
                                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                    Invoice Number
                                  </th>
                                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                    Supplier
                                  </th>
                                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                    Date
                                  </th>
                                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                    Status
                                  </th>
                                </tr>
                              </thead>
                              <tbody className="bg-white divide-y divide-gray-200">
                                {invoices.length === 0 ? (
                                  <tr>
                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500" colSpan={4}>
                                      No invoices found
                                    </td>
                                  </tr>
                                ) : (
                                  invoices.map((invoice) => (
                                    <tr key={invoice.id}>
                                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        {invoice.invoice_number}
                                      </td>
                                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                        {invoice.supplier_name}
                                      </td>
                                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                        {new Date(invoice.invoice_date).toLocaleDateString()}
                                      </td>
                                      <td className="px-6 py-4 whitespace-nowrap">
                                        <span className="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800">
                                          Processed
                                        </span>
                                      </td>
                                    </tr>
                                  ))
                                )}
                              </tbody>
                            </table>
                          </div>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </>
        ) : (
          <div className="bg-white shadow rounded-lg">
            <div className="px-4 py-5 sm:p-6">
              <h2 className="text-lg leading-6 font-medium text-gray-900 mb-4">Review Extracted Data</h2>
              <InvoiceDataGrid
                data={extractedData}
                onSave={handleSaveInvoice}
                onCancel={() => setExtractedData(null)}
              />
            </div>
          </div>
        )}
      </main>
    </div>
  );
}

export default Dashboard;