import React, { useState } from 'react';
import {
  createColumnHelper,
  flexRender,
  getCoreRowModel,
  useReactTable,
} from '@tanstack/react-table';
import { ExtractedInvoiceData } from '../services/ollama';
import { Pencil, Save, X } from 'lucide-react';

interface InvoiceDataGridProps {
  data: ExtractedInvoiceData;
  onSave: (data: ExtractedInvoiceData) => void;
  onCancel: () => void;
}

export default function InvoiceDataGrid({ data, onSave, onCancel }: InvoiceDataGridProps) {
  const [editableData, setEditableData] = useState<ExtractedInvoiceData>(data);
  const [editingCell, setEditingCell] = useState<string | null>(null);

  const columnHelper = createColumnHelper<typeof editableData.items[0]>();

  const columns = [
    columnHelper.accessor('productCode', {
      header: 'Product Code',
      cell: (info) => (
        <EditableCell
          value={info.getValue()}
          isEditing={editingCell === `${info.row.index}-productCode`}
          onEdit={() => setEditingCell(`${info.row.index}-productCode`)}
          onChange={(value) => updateItem(info.row.index, 'productCode', value)}
          onDone={() => setEditingCell(null)}
        />
      ),
    }),
    columnHelper.accessor('description', {
      header: 'Description',
      cell: (info) => (
        <EditableCell
          value={info.getValue()}
          isEditing={editingCell === `${info.row.index}-description`}
          onEdit={() => setEditingCell(`${info.row.index}-description`)}
          onChange={(value) => updateItem(info.row.index, 'description', value)}
          onDone={() => setEditingCell(null)}
        />
      ),
    }),
    columnHelper.accessor('quantity', {
      header: 'Quantity',
      cell: (info) => (
        <EditableCell
          value={info.getValue().toString()}
          isEditing={editingCell === `${info.row.index}-quantity`}
          onEdit={() => setEditingCell(`${info.row.index}-quantity`)}
          onChange={(value) => updateItem(info.row.index, 'quantity', Number(value))}
          onDone={() => setEditingCell(null)}
          type="number"
        />
      ),
    }),
    columnHelper.accessor('unitPrice', {
      header: 'Unit Price',
      cell: (info) => (
        <EditableCell
          value={info.getValue().toString()}
          isEditing={editingCell === `${info.row.index}-unitPrice`}
          onEdit={() => setEditingCell(`${info.row.index}-unitPrice`)}
          onChange={(value) => updateItem(info.row.index, 'unitPrice', Number(value))}
          onDone={() => setEditingCell(null)}
          type="number"
        />
      ),
    }),
  ];

  const table = useReactTable({
    data: editableData.items,
    columns,
    getCoreRowModel: getCoreRowModel(),
  });

  const updateItem = (rowIndex: number, field: string, value: string | number) => {
    setEditableData((prev) => ({
      ...prev,
      items: prev.items.map((item, index) =>
        index === rowIndex ? { ...item, [field]: value } : item
      ),
    }));
  };

  return (
    <div className="space-y-6">
      <div className="grid grid-cols-2 gap-4">
        <div>
          <label className="block text-sm font-medium text-gray-700">Invoice Number</label>
          <EditableCell
            value={editableData.invoiceNumber}
            isEditing={editingCell === 'invoiceNumber'}
            onEdit={() => setEditingCell('invoiceNumber')}
            onChange={(value) => setEditableData({ ...editableData, invoiceNumber: value })}
            onDone={() => setEditingCell(null)}
          />
        </div>
        <div>
          <label className="block text-sm font-medium text-gray-700">Invoice Date</label>
          <EditableCell
            value={editableData.invoiceDate}
            isEditing={editingCell === 'invoiceDate'}
            onEdit={() => setEditingCell('invoiceDate')}
            onChange={(value) => setEditableData({ ...editableData, invoiceDate: value })}
            onDone={() => setEditingCell(null)}
            type="date"
          />
        </div>
        <div>
          <label className="block text-sm font-medium text-gray-700">Supplier Name</label>
          <EditableCell
            value={editableData.supplierName}
            isEditing={editingCell === 'supplierName'}
            onEdit={() => setEditingCell('supplierName')}
            onChange={(value) => setEditableData({ ...editableData, supplierName: value })}
            onDone={() => setEditingCell(null)}
          />
        </div>
        <div>
          <label className="block text-sm font-medium text-gray-700">Supplier Order Number</label>
          <EditableCell
            value={editableData.supplierOrderNumber || ''}
            isEditing={editingCell === 'supplierOrderNumber'}
            onEdit={() => setEditingCell('supplierOrderNumber')}
            onChange={(value) => setEditableData({ ...editableData, supplierOrderNumber: value })}
            onDone={() => setEditingCell(null)}
          />
        </div>
      </div>

      <div className="mt-8 overflow-x-auto">
        <table className="min-w-full divide-y divide-gray-200">
          <thead className="bg-gray-50">
            {table.getHeaderGroups().map((headerGroup) => (
              <tr key={headerGroup.id}>
                {headerGroup.headers.map((header) => (
                  <th
                    key={header.id}
                    className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider"
                  >
                    {flexRender(header.column.columnDef.header, header.getContext())}
                  </th>
                ))}
              </tr>
            ))}
          </thead>
          <tbody className="bg-white divide-y divide-gray-200">
            {table.getRowModel().rows.map((row) => (
              <tr key={row.id}>
                {row.getVisibleCells().map((cell) => (
                  <td key={cell.id} className="px-6 py-4 whitespace-nowrap">
                    {flexRender(cell.column.columnDef.cell, cell.getContext())}
                  </td>
                ))}
              </tr>
            ))}
          </tbody>
        </table>
      </div>

      <div className="flex justify-end space-x-4 mt-4">
        <button
          onClick={onCancel}
          className="px-4 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50"
        >
          Cancel
        </button>
        <button
          onClick={() => onSave(editableData)}
          className="px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700"
        >
          Save Invoice
        </button>
      </div>
    </div>
  );
}

interface EditableCellProps {
  value: string;
  isEditing: boolean;
  onEdit: () => void;
  onChange: (value: string) => void;
  onDone: () => void;
  type?: string;
}

function EditableCell({ value, isEditing, onEdit, onChange, onDone, type = 'text' }: EditableCellProps) {
  if (isEditing) {
    return (
      <div className="flex items-center space-x-2">
        <input
          type={type}
          value={value}
          onChange={(e) => onChange(e.target.value)}
          className="block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm"
          autoFocus
        />
        <button
          onClick={onDone}
          className="p-1 text-gray-400 hover:text-gray-500"
        >
          <Save className="h-4 w-4" />
        </button>
      </div>
    );
  }

  return (
    <div className="flex items-center justify-between group">
      <span>{value}</span>
      <button
        onClick={onEdit}
        className="invisible group-hover:visible p-1 text-gray-400 hover:text-gray-500"
      >
        <Pencil className="h-4 w-4" />
      </button>
    </div>
  );
}